﻿using Cinemachine;
using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using UnityEngine;
using UnityEngine.UI;

public static class Utilities
{
    /// <summary>
    /// Normalizes an input value to either equal -1, 0 or 1
    /// </summary>
    /// <param name="inputValue">The raw input value</param>
    /// <returns></returns>
    public static float NormalizeInput(float inputValue)
    {
        if (inputValue > 0.00f)
        {
            return 1.0f;
        }

        else if (inputValue < 0.00f)
        {
            return -1.0f;
        }

        else
        {
            return 0.0f;
        }
    }

    /// <summary>
    /// Sets a sprite renderer's material alpha (opacity) to the desired opacity value
    /// </summary>
    /// <param name="renderer">The renderer to set the opacity of</param>
    /// <param name="opacity">The opacity value to set</param>
    /// <returns></returns>
    public static Color SetOpacity(this SpriteRenderer renderer, float opacity)
    {
        Color origColour = renderer.material.color;
        renderer.material.color = new Color(origColour.r, origColour.g, origColour.b, opacity);
        return renderer.material.color;
    }

    /// <summary>
    /// Finds a child by name of the parent game object
    /// </summary>
    /// <param name="obj">The parent game object to check for children</param>
    /// <param name="name">The name of the desired child</param>
    /// <returns></returns>
    public static GameObject FindChild(this GameObject obj, string name)
    {
        if (obj != null)
        {
            foreach (Transform child in obj.transform)
            {
                if (child.gameObject.name == name)
                {
                    return child.gameObject;
                }
            }
        }

        return null;
    }

    /// <summary>
    /// Gets all child transforms from the parent transform
    /// </summary>
    /// <param name="parent"></param>
    /// <param name="transformList"></param>
    /// <returns></returns>
    public static List<Transform> GetAllChildren(this Transform parent, List<Transform> transformList = null)
    {
        if (transformList == null)
        {
            transformList = new List<Transform>();
        }

        foreach (Transform child in parent)
        {
            transformList.Add(child);
            child.GetAllChildren(transformList);
        }

        return transformList;
    }

    /// <summary>
    /// Enables the game object and all children by making them interactable
    /// </summary>
    /// <param name="obj">The game object to enable</param>
    public static void Enable(this GameObject obj)
    {
        try
        {
            foreach (Transform child in obj.transform)
            {
                if (child.childCount > 0)
                {
                    child.gameObject.Enable();
                }

                foreach (Component component in child.GetComponents(typeof(Selectable)))
                {
                    ((Selectable)component).interactable = true;
                }
            }
        }

        catch(Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Disables the game object and all children by making them uninteractable
    /// </summary>
    /// <param name="obj">The game object to disable</param>
    public static void Disable(this GameObject obj)
    {
        try
        {
            foreach (Transform child in obj.transform)
            {
                if (child.childCount > 0)
                {
                    child.gameObject.Disable();
                }

                foreach (Component component in child.GetComponents(typeof(Selectable)))
                {
                    ((Selectable)component).interactable = false;
                }
            }
        }

        catch(Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Hides the game object and all children by disabling all renderers and setting the alpha of all canvas renderers to 0
    /// </summary>
    /// <param name="obj">The game object to hide</param>
    public static void Hide(this GameObject obj)
    {
        try
        {
            foreach (Transform child in obj.transform)
            {
                if (child.childCount > 0)
                {
                    child.gameObject.Hide();
                }

                foreach (Component component in child.GetComponents(typeof(Renderer)))
                {
                    ((Renderer)component).enabled = false;
                }

                foreach (Component component in child.GetComponents(typeof(CanvasRenderer)))
                {
                    ((CanvasRenderer)component).SetAlpha(0);
                }
            }
        }

        catch(Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Shows thhe game object and all children by enabling all renderers and setting the alpha of all canvas renderers to 1
    /// </summary>
    /// <param name="obj">The game object to show</param>
    public static void Show(this GameObject obj)
    {
        try
        {
            foreach (Transform child in obj.transform)
            {
                if (child.childCount > 0)
                {
                    child.gameObject.Show();
                }

                foreach (Component component in child.GetComponents(typeof(Renderer)))
                {
                    ((Renderer)component).enabled = true;
                }

                foreach (Component component in child.GetComponents(typeof(CanvasRenderer)))
                {
                    ((CanvasRenderer)component).SetAlpha(1);
                }
            }
        }

        catch (Exception ex)
        {
            Debug.Log("Exception: " + ex);
        }
    }

    /// <summary>
    /// Blends between 2 virtual cameras by setting the priorties
    /// </summary>
    /// <param name="cam">The camera to blend from</param>
    /// <param name="newCam">The camera to blend to</param>
    public static void BlendTo(this CinemachineVirtualCamera cam, CinemachineVirtualCamera newCam)
    {
        cam.Priority = 10;
        newCam.Priority = 99;
        newCam.gameObject.SetActive(true);
    }
}
